class SmartBotApp {
    constructor() {
        this.token = localStorage.getItem('smartBotToken');
        this.currentConversationId = null;
        this.isRecording = false;
        this.mediaRecorder = null;
        this.audioChunks = [];
        
        this.init();
    }
    
    init() {
        this.checkAuth();
        this.bindEvents();
        
        if (this.token) {
            this.loadConversations();
        }
    }
    
    checkAuth() {
        if (!this.token) {
            window.location.href = 'login.html';
            return;
        }
        
        // عرض بريد المستخدم
        const userEmail = localStorage.getItem('smartBotUserEmail');
        if (userEmail) {
            document.getElementById('userEmail').textContent = userEmail;
        }
    }
    
    bindEvents() {
        // أحداث الأزرار
        document.getElementById('logoutBtn').addEventListener('click', () => this.logout());
        document.getElementById('newChatBtn').addEventListener('click', () => this.newConversation());
        document.getElementById('sendBtn').addEventListener('click', () => this.sendMessage());
        document.getElementById('messageInput').addEventListener('keypress', (e) => {
            if (e.key === 'Enter' && !e.shiftKey) {
                e.preventDefault();
                this.sendMessage();
            }
        });
        
        // أحداث التسجيل الصوتي
        document.getElementById('recordBtn').addEventListener('click', () => this.toggleRecording());
        
        // أحداث رفع الملف
        document.getElementById('fileInput').addEventListener('change', (e) => this.handleFileUpload(e));
        
        // أحداث النص المحول
        document.getElementById('sendTranscriptionBtn').addEventListener('click', () => this.sendTranscription());
        document.getElementById('cancelTranscriptionBtn').addEventListener('click', () => this.cancelTranscription());
    }
    
    async loadConversations() {
        try {
            const response = await fetch('api/conversations.php', {
                headers: {
                    'Authorization': `Bearer ${this.token}`
                }
            });
            
            const data = await response.json();
            
            if (data.success) {
                this.displayConversations(data.conversations);
            } else {
                this.showError('فشل في تحميل المحادثات');
            }
        } catch (error) {
            this.showError('خطأ في الاتصال بالخادم');
        }
    }
    
    displayConversations(conversations) {
        const container = document.getElementById('conversationsList');
        container.innerHTML = '';
        
        conversations.forEach(conv => {
            const div = document.createElement('div');
            div.className = 'conversation-item';
            div.innerHTML = `
                <div class="conversation-preview">
                    ${conv.first_message ? this.truncateText(conv.first_message, 50) : 'محادثة جديدة'}
                </div>
                <small>${new Date(conv.created_at).toLocaleDateString('ar-SA')}</small>
            `;
            
            div.addEventListener('click', () => this.loadConversation(conv.id));
            container.appendChild(div);
        });
    }
    
    async loadConversation(conversationId) {
        try {
            const response = await fetch(`api/conversations.php?id=${conversationId}`, {
                headers: {
                    'Authorization': `Bearer ${this.token}`
                }
            });
            
            const data = await response.json();
            
            if (data.success) {
                this.currentConversationId = conversationId;
                this.displayMessages(data.messages);
                
                // تحديث النشاط في قائمة المحادثات
                document.querySelectorAll('.conversation-item').forEach(item => {
                    item.classList.remove('active');
                });
                event.target.closest('.conversation-item').classList.add('active');
            } else {
                this.showError('فشل في تحميل المحادثة');
            }
        } catch (error) {
            this.showError('خطأ في الاتصال بالخادم');
        }
    }
    
    displayMessages(messages) {
        const container = document.getElementById('messagesContainer');
        container.innerHTML = '';
        
        messages.forEach(msg => {
            const messageDiv = document.createElement('div');
            messageDiv.className = `message ${msg.sender}`;
            
            let content = `
                <div class="message-content">${this.escapeHtml(msg.original_text || msg.transcribed_text || '')}</div>
                <div class="message-time">${new Date(msg.created_at).toLocaleString('ar-SA')}</div>
            `;
            
            if (msg.sender === 'user' && msg.media_path) {
                const fileExt = msg.media_path.split('.').pop().toLowerCase();
                if (['mp3', 'wav', 'ogg'].includes(fileExt)) {
                    content += `<div class="media-player"><audio controls src="${msg.media_path}"></audio></div>`;
                } else if (['mp4', 'webm'].includes(fileExt)) {
                    content += `<div class="media-player"><video controls src="${msg.media_path}" width="300"></video></div>`;
                }
            }
            
            if (msg.sender === 'bot' && msg.bot_response) {
                content = `
                    <div class="message-content">${this.escapeHtml(msg.bot_response)}</div>
                    <div class="message-time">${new Date(msg.created_at).toLocaleString('ar-SA')}</div>
                `;
                
                if (msg.response_media_path) {
                    content += `<div class="media-player"><audio controls src="${msg.response_media_path}"></audio></div>`;
                }
            }
            
            messageDiv.innerHTML = content;
            container.appendChild(messageDiv);
        });
        
        container.scrollTop = container.scrollHeight;
    }
    
    async sendMessage() {
        const messageInput = document.getElementById('messageInput');
        const message = messageInput.value.trim();
        
        if (!message) return;
        
        // إضافة الرسالة إلى الواجهة فوراً
        this.addMessageToUI('user', message);
        messageInput.value = '';
        
        try {
            const response = await fetch('api/messages.php?type=text', {
                method: 'POST',
                headers: {
                    'Content-Type': 'application/json',
                    'Authorization': `Bearer ${this.token}`
                },
                body: JSON.stringify({
                    message_text: message
                })
            });
            
            const data = await response.json();
            
            if (data.success) {
                this.addMessageToUI('bot', data.bot_response, data.response_media_path);
                this.currentConversationId = data.conversation_id;
                this.loadConversations(); // تحديث قائمة المحادثات
            } else {
                this.showError(data.message || 'فشل في إرسال الرسالة');
            }
        } catch (error) {
            this.showError('خطأ في الاتصال بالخادم');
        }
    }
    
    async sendMediaMessage(file, type, generateAudio = false) {
        const formData = new FormData();
        formData.append('media', file);
        formData.append('type', type);
        formData.append('generate_audio', generateAudio.toString());
        
        try {
            const response = await fetch('api/messages.php', {
                method: 'POST',
                headers: {
                    'Authorization': `Bearer ${this.token}`
                },
                body: formData
            });
            
            const data = await response.json();
            
            if (data.success) {
                // عرض النص المحول
                this.showTranscription(data.transcribed_text, data.bot_response, data.response_media_path);
                this.currentConversationId = data.conversation_id;
                this.loadConversations();
            } else {
                this.showError(data.message || 'فشل في معالجة الملف');
            }
        } catch (error) {
            this.showError('خطأ في الاتصال بالخادم');
        }
    }
    
    showTranscription(transcribedText, botResponse, responseMediaPath) {
        document.getElementById('transcribedText').textContent = transcribedText;
        document.getElementById('transcriptionArea').classList.remove('hidden');
        
        // حفظ البيانات مؤقتاً للإرسال
        this.pendingTranscription = { transcribedText, botResponse, responseMediaPath };
    }
    
    sendTranscription() {
        if (this.pendingTranscription) {
            this.addMessageToUI('user', this.pendingTranscription.transcribedText);
            this.addMessageToUI('bot', this.pendingTranscription.botResponse, this.pendingTranscription.responseMediaPath);
            this.hideTranscription();
        }
    }
    
    cancelTranscription() {
        this.hideTranscription();
    }
    
    hideTranscription() {
        document.getElementById('transcriptionArea').classList.add('hidden');
        this.pendingTranscription = null;
    }
    
    addMessageToUI(sender, text, mediaPath = null) {
        const container = document.getElementById('messagesContainer');
        
        // إخفاء رسالة الترحيب إذا كانت موجودة
        const welcomeMsg = container.querySelector('.welcome-message');
        if (welcomeMsg) {
            welcomeMsg.remove();
        }
        
        const messageDiv = document.createElement('div');
        messageDiv.className = `message ${sender}`;
        
        let content = `
            <div class="message-content">${this.escapeHtml(text)}</div>
            <div class="message-time">${new Date().toLocaleString('ar-SA')}</div>
        `;
        
        if (mediaPath) {
            content += `<div class="media-player"><audio controls src="${mediaPath}"></audio></div>`;
        }
        
        messageDiv.innerHTML = content;
        container.appendChild(messageDiv);
        container.scrollTop = container.scrollHeight;
    }
    
    async toggleRecording() {
        if (!this.isRecording) {
            await this.startRecording();
        } else {
            this.stopRecording();
        }
    }
    
    async startRecording() {
        try {
            const stream = await navigator.mediaDevices.getUserMedia({ audio: true });
            this.mediaRecorder = new MediaRecorder(stream);
            this.audioChunks = [];
            
            this.mediaRecorder.ondataavailable = (event) => {
                this.audioChunks.push(event.data);
            };
            
            this.mediaRecorder.onstop = () => {
                const audioBlob = new Blob(this.audioChunks, { type: 'audio/wav' });
                this.sendMediaMessage(audioBlob, 'audio', true);
            };
            
            this.mediaRecorder.start();
            this.isRecording = true;
            
            // تحديث الواجهة
            document.getElementById('recordBtn').classList.add('recording');
            document.getElementById('recordingIndicator').classList.remove('hidden');
            document.getElementById('recordBtn').textContent = '⏹ إيقاف';
        } catch (error) {
            this.showError('لا يمكن الوصول إلى الميكروفون');
        }
    }
    
    stopRecording() {
        if (this.mediaRecorder && this.isRecording) {
            this.mediaRecorder.stop();
            this.mediaRecorder.stream.getTracks().forEach(track => track.stop());
            this.isRecording = false;
            
            // تحديث الواجهة
            document.getElementById('recordBtn').classList.remove('recording');
            document.getElementById('recordingIndicator').classList.add('hidden');
            document.getElementById('recordBtn').textContent = '🎤 تسجيل صوت';
        }
    }
    
    handleFileUpload(event) {
        const file = event.target.files[0];
        if (!file) return;
        
        const type = file.type.startsWith('audio/') ? 'audio' : 'video';
        this.sendMediaMessage(file, type, true);
        event.target.value = ''; // إعادة تعيين حقل الرفع
    }
    
    newConversation() {
        this.currentConversationId = null;
        document.getElementById('messagesContainer').innerHTML = `
            <div class="welcome-message">
                <h2>مرحباً في سمارت بوت! 👋</h2>
                <p>يمكنك البدء بمحادثة جديدة أو اختيار محادثة سابقة من القائمة.</p>
            </div>
        `;
        
        // إلغاء تحديد المحادثات
        document.querySelectorAll('.conversation-item').forEach(item => {
            item.classList.remove('active');
        });
    }
    
    logout() {
        localStorage.removeItem('smartBotToken');
        localStorage.removeItem('smartBotUserEmail');
        window.location.href = 'login.html';
    }
    
    showError(message) {
        // يمكن تحسين هذا لعرض رسائل خطأ أكثر جمالاً
        alert(message);
    }
    
    escapeHtml(text) {
        const div = document.createElement('div');
        div.textContent = text;
        return div.innerHTML;
    }
    
    truncateText(text, length) {
        return text.length > length ? text.substring(0, length) + '...' : text;
    }
}

// تهيئة التطبيق عند تحميل الصفحة
document.addEventListener('DOMContentLoaded', () => {
    new SmartBotApp();
});