<?php
require_once 'config.php';

class Auth {
    private $db;
    
    public function __construct($database) {
        $this->db = $database->getConnection();
    }
    
    // توليد JWT token
    public function generateToken($userId, $email) {
        $header = json_encode(['typ' => 'JWT', 'alg' => JWT_ALGORITHM]);
        $payload = json_encode([
            'user_id' => $userId,
            'email' => $email,
            'exp' => time() + (24 * 60 * 60) // 24 ساعة
        ]);
        
        $base64Header = str_replace(['+', '/', '='], ['-', '_', ''], base64_encode($header));
        $base64Payload = str_replace(['+', '/', '='], ['-', '_', ''], base64_encode($payload));
        
        $signature = hash_hmac('sha256', $base64Header . "." . $base64Payload, JWT_SECRET, true);
        $base64Signature = str_replace(['+', '/', '='], ['-', '_', ''], base64_encode($signature));
        
        return $base64Header . "." . $base64Payload . "." . $base64Signature;
    }
    
    // التحقق من JWT token
    public function verifyToken($token) {
        $parts = explode('.', $token);
        if (count($parts) != 3) {
            return false;
        }
        
        list($base64Header, $base64Payload, $base64Signature) = $parts;
        
        $signature = hash_hmac('sha256', $base64Header . "." . $base64Payload, JWT_SECRET, true);
        $calculatedSignature = str_replace(['+', '/', '='], ['-', '_', ''], base64_encode($signature));
        
        if ($calculatedSignature !== $base64Signature) {
            return false;
        }
        
        $payload = json_decode(base64_decode($base64Payload), true);
        
        if (isset($payload['exp']) && $payload['exp'] < time()) {
            return false;
        }
        
        return $payload;
    }
    
    // تسجيل مستخدم جديد
    public function register($email, $password) {
        // التحقق من البريد الإلكتروني
        if (!filter_var($email, FILTER_VALIDATE_EMAIL)) {
            return ['success' => false, 'message' => 'البريد الإلكتروني غير صالح'];
        }
        
        // التحقق من كلمة المرور
        if (strlen($password) < 6) {
            return ['success' => false, 'message' => 'كلمة المرور يجب أن تكون 6 أحرف على الأقل'];
        }
        
        // التحقق من عدم وجود البريد مسبقاً
        $stmt = $this->db->prepare("SELECT id FROM users WHERE email = ?");
        $stmt->execute([$email]);
        
        if ($stmt->fetch()) {
            return ['success' => false, 'message' => 'البريد الإلكتروني مسجل مسبقاً'];
        }
        
        // إنشاء المستخدم
        $passwordHash = password_hash($password, PASSWORD_DEFAULT);
        $stmt = $this->db->prepare("INSERT INTO users (email, password_hash) VALUES (?, ?)");
        
        if ($stmt->execute([$email, $passwordHash])) {
            $userId = $this->db->lastInsertId();
            $token = $this->generateToken($userId, $email);
            
            return [
                'success' => true,
                'user_id' => $userId,
                'token' => $token
            ];
        }
        
        return ['success' => false, 'message' => 'خطأ في إنشاء الحساب'];
    }
    
    // تسجيل الدخول
    public function login($email, $password) {
        $stmt = $this->db->prepare("SELECT id, email, password_hash FROM users WHERE email = ?");
        $stmt->execute([$email]);
        $user = $stmt->fetch();
        
        if (!$user || !password_verify($password, $user['password_hash'])) {
            return ['success' => false, 'message' => 'البريد الإلكتروني أو كلمة المرور غير صحيحة'];
        }
        
        $token = $this->generateToken($user['id'], $user['email']);
        
        return [
            'success' => true,
            'user_id' => $user['id'],
            'token' => $token
        ];
    }
    
    // الحصول على بيانات المستخدم من التوكن
    public function getUserFromToken($token) {
        $payload = $this->verifyToken($token);
        if (!$payload) {
            return null;
        }
        
        $stmt = $this->db->prepare("SELECT id, email FROM users WHERE id = ?");
        $stmt->execute([$payload['user_id']]);
        
        return $stmt->fetch();
    }
}
?>